/* LDBS: LibDsk Block Store access functions
 *
 *  Copyright (c) 2016-17 John Elliott <seasip.webmaster@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the 
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included 
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR 
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
 * OTHER DEALINGS IN THE SOFTWARE. */

/* LDBS example software: Upgrade an LDBS file to the latest version. */

#define _CRT_SECURE_NO_WARNINGS
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ldbs.h"

void diewith(const char *s, dsk_err_t err)
{
	switch (err)
	{
		case DSK_ERR_OK:
			break;
		case DSK_ERR_NOTME: 
			fprintf(stderr, "%s: File is not in LDBS disk format\n", s); 
			break;
		case DSK_ERR_SYSERR:
			perror(s);
			break;
		case DSK_ERR_NOMEM:
			fprintf(stderr, "Out of memory\n");
			break;
		default:
			fprintf(stderr, "%s: LibDsk error %d\n", 
				s, err);
			break;
	}
	exit(1);
}



/* We can't do an ldbs_clone() because that will force the destination to 
 * be in the same version as the source. So instead of copying all the blocks
 * and then remapping them, just copy blocks mentioned in the track directory. 
 */
static dsk_err_t ldbs_to_ldbs(PLDBS source, PLDBS dest)
{
	LDBS_TRACKDIR *srcdir;
	dsk_err_t err;
	unsigned n, ns;
	unsigned char *buf1;
	size_t len1;
	dsk_pcyl_t cyl;
	dsk_phead_t head;

	/* Load the trackdir from the source file */

	err = ldbs_trackdir_copy(source, &srcdir);
	if (err) return err;

	for (n = 0; n < srcdir->count; n++)
	{
		if (srcdir->entry[n].id[0] == 'T')	/* Track header? */
		{
			LDBS_TRACKHEAD *trkh;

			ldbs_decode_trackid(srcdir->entry[n].id, &cyl, &head);
			err = ldbs_get_trackhead(source, &trkh, cyl, head);
			if (err) break;

			for (ns = 0; ns < trkh->count; ns++)
			{
				char type[4];

				if (trkh->sector[ns].blockid)
				{
					err = ldbs_getblock_a(source, 
						trkh->sector[ns].blockid,
						type, (void **)&buf1, &len1);
					if (err) break;
					trkh->sector[ns].blockid = LDBLOCKID_NULL;
					err = ldbs_putblock(dest, 
						&trkh->sector[ns].blockid,
						type, buf1, len1);
					ldbs_free(buf1);
					if (err) break;
				}
			}
			if (!err)
			{
				err = ldbs_put_trackhead(dest, trkh, cyl, head);
			}
			ldbs_free(trkh);
			if (err) break;
		}
		else
		{
			err = ldbs_getblock_a(source, srcdir->entry[n].blockid,
				NULL, (void **)&buf1, &len1);
			if (err) break;
			err = ldbs_putblock_d(dest, srcdir->entry[n].id,
					buf1, len1);
			ldbs_free(buf1);
			if (err) break;
		}
	}
	ldbs_free(srcdir);
	return err;	
}


int main(int argc, char **argv)
{
	int ro;
	dsk_err_t err;
	PLDBS source, dest;
	char type[4];

	if (argc < 3)
	{
		fprintf(stderr, "%s: Syntax is %s infile outfile\n",
				argv[0], argv[0]);
		exit(1);
	}
	ro = 1;
	err = ldbs_open(&source, argv[1], type, &ro);
	if (err) diewith(argv[1], err);

	if (!memcmp(type, LDBS_DSK_TYPE, 4))
	{
		fprintf(stderr, "File %s is already in LDBS v%s format",
			argv[1], LDBS_VERSION);
		ldbs_close(&source);
		return 0;
	}

	if (memcmp(type, LDBS_DSK_TYPE_V1, 4))
	{
		ldbs_close(&source);
		diewith(argv[1], DSK_ERR_NOTME);
	}
	err = ldbs_new(&dest, argv[2], LDBS_DSK_TYPE);
	if (err)
	{
		ldbs_close(&source);
		diewith(argv[2], err);
	}
	/* Sadly we can't just use the builtin ldbs_clone() because it will 
	 * force the destination to be the same version as the source. So
	 * instead, we have to do a more manual copy. */
	err = ldbs_to_ldbs(source, dest);
	if (err)
	{
		ldbs_close(&dest);
		ldbs_close(&source);
		diewith("Clone", err);
	}
	err = ldbs_close(&dest);
	if (err)
	{
		ldbs_close(&source);
		diewith(argv[2], err);
	}
	ldbs_close(&source);
	return 0;
}
