/***************************************************************************
 *                                                                         *
 *    DISK2ROM: Convert a floppy image to a Toshiba 1000 boot ROM          *
 *    Copyright (C) 2017 John Elliott <seasip.webmaster@gmail.com>         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or modify *
 *    it under the terms of the GNU General Public License as published by *
 *    the Free Software Foundation; either version 2 of the License, or    *
 *    (at your option) any later version.                                  *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *    along with this program; if not, write to the Free Software          *
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA        *
 *    02110-1301 USA.                                                      *
 *                                                                         *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libdsk.h"	/* libdsk will be used for drive / disc image access */

#ifdef __PACIFIC__
#define AV0 "DISK2ROM"	/* Pacific C doesn't provide argv[0] */
#else
#define AV0 argv[0]
#endif

#ifndef SEEK_SET	/* Pacific C doesn't provide SEEK_* */
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2
#endif

unsigned char header[1024];	/* ROM drive header */
unsigned char bootsec[512];	/* Boot sector */
unsigned char *fat;		/* FAT */
unsigned char *rootdir;		/* Root directory */

unsigned long romsize = 512;	/* Size of output ROM, kilobytes */
unsigned char secbuf[512];	/* Current sector */
/* Drive geometry... */
unsigned short total_sectors;		/* Sectors on disk */
unsigned short fats;			/* Count of FATs */
unsigned short reserved;		/* Count of reserved sectors */
unsigned short dirents;			/* Count of directory entries */
unsigned short dirsecs;			/* Count of directory sectors */
unsigned short clusters;		/* Total clusters */
unsigned short secs_per_cluster;	/* Sectors per cluster */
unsigned short secs_per_fat;		/* Sectors per FAT */
unsigned short maxclus;			/* Number of usable clusters in ROM */

char *infile = NULL, *outfile = NULL;	/* Filenames of input / output files */
char *itype = NULL;			/* Type of input disk image */
char *icomp = NULL;			/* Compression of input disk image */

FILE *fp;			/* The ROM image */


/* Given a cluster, calculate which logical sector it is located in. */
dsk_lsect_t cluster_lsec(unsigned short cluster)
{
	return reserved + (secs_per_fat * fats) + dirsecs + 
		(secs_per_cluster * (cluster - 2));
}

/* Save a logical sector as cylinder / head / sector in the ROM header */
void store_sector(DSK_GEOMETRY *dg, dsk_lsect_t sec, unsigned char *dest)
{
	dsk_psect_t psec;
	dsk_phead_t phead;
	dsk_pcyl_t pcyl;

	dg_ls2ps(dg, sec, &pcyl, &phead, &psec);

	dest[0] = phead;
	dest[1] = psec;
	dest[2] = pcyl;
}


/* Get the FAT entry for a specified cluster */
unsigned short get_fat(unsigned short cluster)
{
	unsigned offset = (cluster / 2) * 3;

	if (cluster & 1)
	{
		return (fat[offset + 2] << 4) | (fat[offset+1] >> 4);
	}
	else
	{
		return fat[offset] | ((fat[offset + 1] & 0x0F) << 8);
	}
}

/* Read a 16-bit little-endian word */
unsigned short peek2(unsigned char *addr)
{
	return addr[0] + 256 * addr[1];
}

/* Display syntax */
void syntax(char *av0)
{
	fprintf(stderr, "Syntax: %s { options } disk_image rom_image\n\n", av0);
	fprintf(stderr, "Options:\n"
			"  -128            Output a 128k ROM image\n"
			"  -256            Output a 256k ROM image\n"
			"  -512            Output a 512k ROM image (default)\n"
			"  -type typename  Format of input file, eg 'raw' or 'imd'\n"
			"  -comp compname  Compression used on input file.\n");
}


/* And, finally, the body of the program */
int main(int argc, char **argv)
{
	int n;
	int endopt = 0;
	int c;
	long pos;
	DSK_PDRIVER dsk;
	DSK_GEOMETRY geom;
	dsk_err_t err;
	dsk_lsect_t sec;
	int badformat = 0;
	unsigned short cksum;

	/* Parse command line arguments */
	for (n = 1; n < argc; n++)
	{
		if (!strcmp(argv[n], "--"))
		{
			endopt = 1; 
			continue;
		}
		if (!endopt)
		{
			if (!strcmp(argv[n], "/?") || !strcmp(argv[n], "/H") ||
			    !strcmp(argv[n], "/h") || !strcmp(argv[n], "-h") ||
			    !strcmp(argv[n], "-help") ||
			    !strcmp(argv[n], "--help"))
			{
				syntax(AV0);
				return 0;
			}
			else if (argv[n][0] == '-')
			{
				/* Check for ROM size options */
				if (!strcmp(argv[n], "-512")) 
				{
					romsize = 512;
					continue;
				}
				if (!strcmp(argv[n], "-256")) 
				{
					romsize = 256;
					continue;
				}
				if (!strcmp(argv[n], "-128")) 
				{
					romsize = 128;
					continue;
				}
				/* Check for libdsk options */
				if ((!strcmp(argv[n], "-itype") || 
				     !strcmp(argv[n], "-type")) && (n+1) < argc)
				{
					++n;
					itype = argv[n];
					continue;
				}
				if ((!strcmp(argv[n], "-icomp") || 
				     !strcmp(argv[n], "-comp")) && (n+1) < argc)
				{
					++n;
					icomp = argv[n];
					continue;
				}
				fprintf(stderr, "Unknown option: %s\n", argv[n]);
				return 1;
			}
		}
		if (!infile) infile = argv[n];
		else if (!outfile) outfile = argv[n];
		else { syntax(AV0); return 1; }
	}
	if (!infile || !outfile)
	{
		syntax(AV0);
		return 1;
	}
	/* Open input file */
	err = dsk_open(&dsk, infile, itype, icomp);
	/* And try to get its geometry from the boot sector */
	if (!err) err = dsk_getgeom(dsk, &geom);
	if (!err)
	{
		/* Do a sanity check: Is there likely to be a 512-byte
		 * boot sector where we are looking for it? */
		if (geom.dg_secsize != 512 || geom.dg_sectors != 9 ||
		    geom.dg_secbase != 1)
		{
			badformat = 1;
		}
		else
		{
			/* Load the boot sector. Check that there is a 
			 * a reasonable-looking BPB on it (libdsk 
			 * provides dg_dosgeom() for this purpose) */
			err = dsk_lread(dsk, &geom, bootsec, 0);
			if (!err)
			{
				DSK_GEOMETRY g2;
	
				if (dg_dosgeom(&g2, bootsec) ||
					g2.dg_cylinders != geom.dg_cylinders ||
					g2.dg_heads     != geom.dg_heads ||
					g2.dg_sectors   != geom.dg_sectors)
				{
					badformat = 1;
				}	
			}
		}
	}
	if (!err && !badformat)	/* Boot sector loaded. Work out the 
				 * FAT filesystem layout */
	{
		secs_per_cluster = bootsec[0x0D];
		reserved      = peek2(bootsec + 0x0E);
		fats	      = bootsec[0x10];
		dirents       = peek2(bootsec + 0x11);
		total_sectors = peek2(bootsec + 0x13);
		secs_per_fat  = peek2(bootsec + 0x16);
		dirsecs       = (dirents + 15) / 16;	/* Directory sectors */

		clusters      = (total_sectors - reserved - dirsecs 
				- (secs_per_fat * fats)) / secs_per_cluster;

		/* Load the FAT */
		fat = malloc(secs_per_fat * 512);
		if (fat && !err)
		{
			for (sec = 0; sec < secs_per_fat; sec++)
			{
				err = dsk_lread(dsk, &geom, fat + 512 * sec, 
					sec + reserved);
				if (err) break;
			}	
		}
		else err = DSK_ERR_NOMEM;

		/* Load the root directory */
		rootdir = malloc(dirsecs * 512);
		if (rootdir && !err)
		{
			for (sec = 0; sec < dirsecs; sec++)
			{
				err = dsk_lread(dsk, &geom, rootdir + 512 * sec,
					sec + reserved + (fats * secs_per_fat));
				if (err) break;
			}
		}
	}
	if (!err && !badformat)	/* Boot sector, FAT and root directory */
	{			/* all loaded. */
		/* Work out how much space we have to play with. */
		unsigned long available = 1024L * (romsize - 1);

		available -= (reserved * 512);	/* Deduct boot sector */
		available -= (fats * secs_per_fat * 512);	/* Deduct FATs*/
		available -= (dirsecs * 512);	/* Deduct root directory */

		/* Number of usable clusters in ROM disk */
		maxclus = (available / (512 * secs_per_cluster));

		/* Start generating the header */
		memset(header, 0xFF, sizeof(header));
		header[0] = (1024 * romsize) & 0xFF;
		header[1] = ((1024 * romsize) >> 8) & 0xFF;
		header[2] = ((1024 * romsize) >> 16) & 0xFF;

		/* Scan the root directory for files that fall outside the 
		 * saved area. On the original Toshiba drive, this is the
		 * dummy file DUM */
		for (c = 0; c < dirents; c++)
		{
			int alert = 0;
			unsigned short cluster;
			unsigned char *dirent = rootdir + 32 * c;
	
			if (dirent[0] == 0xE5) continue; /* Deleted file */
			if (dirent[0] == 0) break;	 /* End of directory */
			if (dirent[11] & 8) continue;	/* Volume label */

 			cluster = peek2(dirent + 0x1A);
/* If we encounter CONFIG.SYS or AUTOEXEC.BAT, update their location in 
 * the ROM header */	
			if (!memcmp(dirent, "CONFIG  SYS", 11) && 
				(dirent[11] & 0x18) == 0)
			{
				sec = cluster_lsec(cluster);
				store_sector(&geom, sec, header + 3);
			}
			if (!memcmp(dirent, "AUTOEXECBAT", 11) &&
				(dirent[11] & 0x18) == 0)
			{
				sec = cluster_lsec(cluster);
				store_sector(&geom, sec, header + 6);
			}
			while (cluster >= 2 && cluster < 0xFF0)
			{
				if ((cluster - 2) > maxclus) alert = 1;
				cluster = get_fat(cluster);	
			}

			if (alert) 
			{
				fprintf(stderr, "Warning: File %-8.8s.%-3.3s falls outside the ROM drive area.\n", dirent, dirent + 8);
			}
		}
	}
	if (badformat)
	{
		fprintf(stderr, "Unsuitable disk format. Input file must "
				"be a disc image of a FAT12 disc with 512\n"
				"sectors per track.\n"); 
		if (dsk) dsk_close(&dsk);
		return 1;
	}
	if (err)
	{
		fprintf(stderr, "%s: %s\n", infile, dsk_strerror(err));
		if (dsk) dsk_close(&dsk);
		return 1;	
	}
	fp = fopen(outfile, "wb");
	if (!fp)
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Cannot open file: %s\n", outfile);
#else
		perror(outfile);
#endif
		if (dsk) dsk_close(&dsk);
		return 1;	
	}
	/* Write the header */
	if (fwrite(header, 1, sizeof(header), fp) < (int)sizeof(header))
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Error writing to file: %s\n", outfile);
#else
		perror(outfile);
#endif
		fclose(fp);
		remove(outfile);
		if (dsk) dsk_close(&dsk);
		return 1;	
	}
	/* Copy the data */
	sec = 0;
	cksum = 0;
	pos = 1024;
	while (pos < 1024L * romsize)
	{
		err = dsk_lread(dsk, &geom, secbuf, sec);
		if (err) break;
		for (c = 0; c < 512; c++) cksum += secbuf[c];	
		if (fwrite(secbuf, 1, 512, fp) < 512)
		{
#ifdef __PACIFIC__
			fprintf(stderr, "Error writing to file: %s\n", outfile);
#else
			perror(outfile);
#endif
			fclose(fp);
			remove(outfile);
			if (dsk) dsk_close(&dsk);
			return 1;	
		}
		pos += 512;
		++sec;	
	}
	/* We don't need anything else from the input file, so close it */
	dsk_close(&dsk);

	/* Write back the checksum */
	if (fseek(fp, 9, SEEK_SET) ||
	    fputc(cksum >> 8, fp) == EOF ||
	    fputc(cksum & 0xFF, fp) == EOF)
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Error writing to file: %s\n", outfile);
#else
		perror(outfile);
#endif
		fclose(fp);
		remove(outfile);
		return 1;	
	}
	/* And close the outputfile */
	if (fclose(fp))
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Error closing file: %s\n", outfile);
#else
		perror(outfile);
#endif
		remove(outfile);
		return 1;	
	}
	return 0;
}
