/***************************************************************************
 *                                                                         *
 *    ROM2DISK: Convert a Toshiba 1000 boot ROM to a floppy image          *
 *    Copyright (C) 2017 John Elliott <seasip.webmaster@gmail.com>         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or modify *
 *    it under the terms of the GNU General Public License as published by *
 *    the Free Software Foundation; either version 2 of the License, or    *
 *    (at your option) any later version.                                  *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *    along with this program; if not, write to the Free Software          *
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA        *
 *    02110-1301 USA.                                                      *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libdsk.h"	/* libdsk will be used for drive / disc image access */

#ifdef __PACIFIC__
#define AV0 "ROM2DISK"	/* Pacific C doesn't provide argv[0] */
#else
#define AV0 argv[0]
#endif

#ifndef SEEK_SET
#define SEEK_SET 0	/* Pacific C doesn't provide SEEK_* */
#define SEEK_CUR 1
#define SEEK_END 2
#endif

unsigned char header[1536];	/* 1k ROM header + 512 bytes boot sector */
unsigned long romsize;		/* Determined size of ROM */
unsigned char sector[512];	/* Current sector */

char *infile = NULL, *outfile = NULL;	/* Filenames of input / output files */
char *otype = "raw";			/* Type of output disk image */
char *ocomp = NULL;			/* Compression of output disk image */

FILE *fp;			/* The ROM image */

/* Read a sector from the ROM drive into sector */
dsk_err_t get_sector(DSK_GEOMETRY *geom, dsk_pcyl_t cyl, dsk_phead_t head,
		dsk_psect_t sec)
{
	long offset = ((cyl * geom->dg_heads) + head) * geom->dg_sectors;

	offset += sec;
	offset *= 512;
	offset += 1024;

	if (offset >= romsize) return DSK_ERR_NOADDR;

	if (fseek(fp, offset, SEEK_SET)) 
	{
		return DSK_ERR_SYSERR;	
	}
	if (fread(sector, 1, sizeof(sector), fp) < (int)sizeof(sector))
	{
		return DSK_ERR_SYSERR;	
	}
	return DSK_ERR_OK;
}

/* Helpscreen */
void syntax(char *av0)
{
	fprintf(stderr, "Syntax: %s { options } rom_image disk_image\n\n", av0);
	fprintf(stderr, "Options:\n"
			"  -type typename  Format of output file, eg 'raw' or 'imd'\n"
			"  -comp compname  Compress output file.\n");
}


/* Display an 'input file in bad format' message */
void badformat(const char *infile, const char *why)
{
	fprintf(stderr, "%s is not in Toshiba ROMdrive format (%s).\n", 
				infile, why);
}

/* And, finally, the body of the program */
int main(int argc, char **argv)
{
	int n;
	int endopt = 0;
	int c;
	long pos;
	unsigned short cksum = 0;
	DSK_PDRIVER dsk;
	DSK_GEOMETRY geom;
	dsk_err_t err;
	dsk_pcyl_t cyl;
	dsk_phead_t head;
	dsk_psect_t sec;

	/* Parse command line arguments */
	for (n = 1; n < argc; n++)
	{
		if (!strcmp(argv[n], "--"))
		{
			endopt = 1; 
			continue;
		}
		if (!endopt)
		{
			if (!strcmp(argv[n], "/?") || !strcmp(argv[n], "/H") ||
			    !strcmp(argv[n], "/h") || !strcmp(argv[n], "-h") ||
			    !strcmp(argv[n], "-help") ||
			    !strcmp(argv[n], "--help"))
			{
				syntax(AV0);
				return 0;
			}
			else if (argv[n][0] == '-')
			{
				/* Check for libdsk options */
				if ((!strcmp(argv[n], "-otype") || 
				     !strcmp(argv[n], "-type")) && (n+1) < argc)
				{
					++n;
					otype = argv[n];
					continue;
				}
				if ((!strcmp(argv[n], "-ocomp") || 
				     !strcmp(argv[n], "-comp")) && (n+1) < argc)
				{
					++n;
					ocomp = argv[n];
					continue;
				}
				fprintf(stderr, "Unknown option: %s\n", argv[n]);
				return 1;
			}
		}
		if (!infile) infile = argv[n];
		else if (!outfile) outfile = argv[n];
		else { syntax(AV0); return 1; }
	}
	if (!infile || !outfile)
	{
		syntax(AV0);
		return 1;
	}
	/* Open the ROM image */
	fp = fopen(infile, "rb");
	if (!fp)
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Cannot open file: %s\n", infile);
#else
		perror(infile);
#endif
		return 1;
	}
	/* Read header and boot sector */
	if (fread(header, 1, sizeof(header), fp) != (int)sizeof(header))
	{
		badformat(infile, "no header");
		fclose(fp);
		return 1;
	}
	romsize = header[0] + 256 * header[1] + 65536 * header[2];
	if (romsize < 1024)
	{
		badformat(infile, "no size given");
		fclose(fp);
		return 1;
	}
	if (fseek(fp, 1024, SEEK_SET))
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Cannot seek in file: %s\n", infile);
#else
		perror(infile);
#endif
		fclose(fp);
		return 1;
	}
	/* Verify the ROM checksum */
	for (pos = 1024; pos < romsize; pos++)
	{
		c = fgetc(fp);
		if (c == EOF)
		{
			badformat(infile, "too short");
			fclose(fp);
			return 1;
		}
		cksum += c;
	}
	if (header[0x09] != (cksum >> 8) || header[0x0A] != (cksum & 0xFF))
	{
		badformat(infile, "bad checksum");
		fclose(fp);
		return 1;
	}
	/* Seek back to the data area */
	if (fseek(fp, 1024, SEEK_SET))
	{
#ifdef __PACIFIC__
		fprintf(stderr, "Cannot seek in file: %s\n", infile);
#else
		perror(infile);
#endif
		fclose(fp);
		return 1;
	}
	/* Ready to create output */
	err = dsk_creat(&dsk, outfile, otype, ocomp);
	if (err)
	{
		fprintf(stderr, "%s: %s\n", outfile, dsk_strerror(err));
		fclose(fp);
		return 1;
	}
	/* Decode the DOS BPB that had just better be at the start of the
	 * ROM. */
	err = dg_dosgeom(&geom, &header[1024]);
	if (err)
	{
		fprintf(stderr, "%s: %s\n", outfile, dsk_strerror(err));
		dsk_close(&dsk);
		remove(outfile);
		fclose(fp);
		return 1;
	}
	/* Then, for each track... */
	for (cyl = 0; cyl < geom.dg_cylinders; cyl++)
	{
		for (head = 0; head < geom.dg_heads; head++)
		{
			/* Format the track */
			err = dsk_apform(dsk, &geom, cyl, head, 0xF6);
			if (err) break;
			for (sec = 0; sec < geom.dg_sectors; sec++)
			{
				/* Load each sector */
				err = get_sector(&geom, cyl, head, sec);
				if (err == DSK_ERR_NOADDR)
				{
					memset(sector, 0xF6, sizeof(sector));
					err = DSK_ERR_OK;
				}
				if (err) break;
				/* And write that sector to the disk image */
				err = dsk_pwrite(dsk, &geom, sector, cyl, head, sec + geom.dg_secbase);
				if (err) break;
			}
			if (err) break;
		}
		if (err) break;
	}
	if (!err) err = dsk_close(&dsk);
	if (err)
	{
		fprintf(stderr, "%s: %s\n", outfile, dsk_strerror(err));
		if (dsk) dsk_close(&dsk);
		remove(outfile);
	}	
	fclose(fp);

	return 0;
}
