/*
    HEXPAT: Patch an object file with an Intel Hex overlay
    Copyright (C) 2004, John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    This program is based on the public domain LOAD.C from the Simtel-20
    UNIX-C collection. Its original author is unknown.  

    LOAD.C
	Convert a .HEX format file to a .COM format file.
	Contributor: unknown

   By this stage, my code outnumbers the original code by a factor of about 3.

*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#define BASE 0x8000		/* JSW default base address */

#ifdef CPM
 #define AV0 "hexpat"
#else
 #ifdef MSDOS
  #define AV0 "hexpat"
 #else
  #define AV0 argv[0]
 #endif
#endif

typedef unsigned char byte;
typedef unsigned short word;


unsigned int base = BASE;	/* Assumed load address of file being patched. */
static byte hex_checksum, tap_checksum;
static word tap_bytes_left;
static int hexline = 0;
static int tapmode = 0;

static FILE *fpout = NULL, *fpcom = NULL, *fphex = NULL;
static byte tap_header[19];


void check_fputc(byte b)
{
	if (fputc(b, fpout) == EOF)
	{
		fprintf(stderr, "Error writing output file.\n");
		exit(3);
	}
}


void writeout(byte b)
{
	check_fputc(b);
	if (tapmode)
	{
		tap_checksum ^= b;
		--tap_bytes_left;
		if (tap_bytes_left == 1) /* Last byte is checksum */
		{
			fgetc(fpcom);	/* Skip old checksum */
			check_fputc(tap_checksum);
		}
	}
}

void fwriteout(byte *b, int len)
{
	while (len--) 
	{
		writeout(*b);
		++b;
	}
}

int decant(FILE *fp, int n)
{
	int ch = 0;

	while (n--)
	{
		ch = fgetc(fp);
		if (ch == EOF)
		{
			fprintf(stderr, "Unexpected end-of-file on input\n");
			exit(3);
		}
		check_fputc(ch);
	}
	return ch;
}



byte gethex(FILE *fp) 
{
	register int    c;
	byte   x;

	c = fgetc(fphex);
	if ('0' <= c && c <= '9') 	x = c - '0';
	else if ('A' <= c && c <= 'F')	x = c - 'A' + 10;
	else 
	{
		fprintf (stderr, "Funny hex letter %c\n", c);
		exit (2);
	}

	x <<= 4;
	c = fgetc(fphex);
	if ('0' <= c && c <= '9') 	x |= c - '0';
	else if ('A' <= c && c <= 'F')	x |= c - 'A' + 10;
	else 
	{
		fprintf (stderr, "Funny hex letter %c\n", c);
		exit (2);
	}
	hex_checksum += x;
	return x;
}

word read_word(FILE *fp)
{
	int x, y;
	x = fgetc(fp);
	y = fgetc(fp);
	if (x == EOF || y == EOF)
	{
		fprintf(stderr, "Premature EOF on .TAP file\n");
		exit(3);
	}
	check_fputc(x);
	check_fputc(y);
	return (y << 8) | x;
}


void load_header(FILE *fp)
{
	word hdrlen;
	int c;
	do
	{
		hdrlen = read_word(fp);
		if (hdrlen == 19) 
		{
			if (fread(tap_header, 1, hdrlen, fp) < hdrlen)
			{
				fprintf(stderr, "Premature EOF on .TAP file\n");
				exit(3);
			}
			fwriteout(tap_header, hdrlen);
			if (tap_header[0] != 0) continue; /* Not a BASIC header */
			switch(tap_header[1])
			{
				case 0: fprintf(stderr, "Program: %-10.10s\n", tap_header + 2);
					break;
				case 1:
				case 2: fprintf(stderr, "Array:   %-10.10s\n", tap_header + 2);
					break;
				case 3: fprintf(stderr, "Bytes:   %-10.10s\n", tap_header + 2);
					break;
			}
		}
		else
		{
			decant(fp, hdrlen);
			continue;
		}
	} while (hdrlen != 19   || 
                 tap_header[0] != 0 || 
                 tap_header[1] != 3);

	tap_bytes_left = read_word(fp);
	c = fgetc(fp);
	writeout(c);
}




int istrcmp(char *s, char *t)
{
	while (*s || *t)
	{
		int d = toupper(*s) - toupper(*t);
		if (d) return d;
		++s;
		++t;
	}
	return 0;

}


void syntax(char *av0)
{
	fprintf(stderr, "Syntax: %s { options } inputfile { outputfile { hexfile })\n",
		av0);	
	fprintf(stderr,"\nOptions are:\n"
		" -com:   File being patched is a CP/M COMmand file\n"
		" -sna:   File being patched is a Spectrum snapshot\n"
		" -tap:   File being patched is a Spectrum .TAP tapefile\n"
		" -bnnnn: Set base address of file being patched.\n"
		"         eg: -b7F80 for a file that loads at 7F80h\n");
	fprintf(stderr,"\nExample:\n");
	fprintf(stderr, "%s -sna jetset.sna jetmono.sna < mono.hex\n", av0); 
	fprintf(stderr,"\nIf outfile and hexfile are missing, stdout and stdin\n"
                "(respectively) will be used.");
	exit(1);
}



int main (int argc, char **argv) 
{
	register unsigned   i, n;
	char    c;
	byte	buf[64];
	int     j;
	unsigned    type;
	unsigned int al, ah, naddr, addr;
	char *fnin = NULL, *fnout = NULL, *fnhex = NULL;

	hexline = 0;

	for (j = 1; j < argc; j++)
	{
		if (argv[j][0] != '-')
		{
			if (fnin == NULL) 
			{
				fnin = argv[j];
				continue;
			}
			if (fnout == NULL)
			{
				fnout = argv[j];
				continue;
			}
			if (fnhex == NULL)
			{
				fnhex = argv[j];
				continue;
			}
			syntax(AV0);
		}
		if (!istrcmp(argv[j], "-sna")) base = 0x3FE5;		
		if (!istrcmp(argv[j], "-com")) base = 0x100;
		if (!istrcmp(argv[j], "-tap")) tapmode = 1;
		if (argv[j][1] != 'b' && argv[j][1] != 'B') continue;
		sscanf(&argv[j][2], "%x", &base);
	}
	if (fnin == NULL) 
		syntax(AV0);	/* Must at least have the input filename */
	else	fpcom = fopen(fnin, "rb");
	if (!fpcom) { perror(fnin); return 1; }

	if (fnhex == NULL)
		fphex = stdin;
	else	fphex = fopen(fnhex, "r");
	if (!fphex) { perror(fnhex); return 1; }

	if (fnout == NULL)
		fpout = stdout;
	else	fpout = fopen(fnout,"wb");
	if (!fpout) { perror(fnout); return 1; }

	if (tapmode)
	{
		load_header(fpcom);		
	}


	addr = base;
	do {
/* Skip until the first hex record starts. Some .HEX files have a line
 * explaining what they are. */
		do {
	 		c = fgetc(fphex);
	    		if (c == EOF) 
			{
				fprintf (stderr, "Premature EOF; colon missing\n");
				exit (1);
	    		}
		} while (c != ':');

		++hexline;
		hex_checksum = 0;
		n  = gethex(fphex);		/* bytes / line */
		ah = gethex(fphex);
		al = gethex(fphex);

	
		switch (type = gethex(fphex)) 
		{
		    case 0:
			if (!n)	/* MAC uses a line with no bytes as EOF */
			{
				type = 1;
				break;
			}
			naddr = (ah << 8) | al;
/* Skip bytes until we reach the desired address */
			while (addr < naddr)  
			{
				j = fgetc(fpcom);
				if (j == EOF) writeout(0);
				else	      writeout(j);
				++addr;
			}
			if (addr > naddr) 
			{
				fprintf(stderr,"Line %d: Records out of sequence at %x > %x\n", hexline, naddr, addr);
				exit(1);
			}

			for (i = 0; i < n; i++)
			{
				/* Step through the binary file */
				(void)fgetc(fpcom);
				buf[i] = gethex(fphex);
			}
			fwriteout(buf, n);
			break;

			case 1: break; 
			default:
				fprintf (stderr, "Line %d: Funny record type %d\n", hexline, type);
				exit (1);
		}

		(void) gethex(fphex);	/* Checksum byte */
		if (hex_checksum != 0) 
		{
		    fprintf (stderr, "Line %d: Checksum error", hexline);
		    exit (2);
		}
		addr += n;
	
	} while (type != 1);

	j = fgetc(fpcom);
	while (j != EOF)
	{
		writeout(j);
		j = fgetc(fpcom);
	}
	if (fpout != stdout)
	{
		if (fclose(stdout)) perror(fnout);
	}
	if (fphex != stdin) fclose(fphex);
	fclose(fpcom);
	return 0;
}
